#!/bin/bash
#
# log.sh - bash functions to manage logfiles
#
# __copy1__
# __copy2__
#
# v1.15 (2022-03-25)
#
export LOGFILE LOGTAG LOGSIZE LOGUSER LOGGROUP LOGMODE LOGSYSLOG
export LOGGERPIO LOGGERARGS LOGROTATIONS
export VERBOSE KU_LOG_SAVEIFS

LOGSIZE=${LOGSIZE:-10240}
LOGFILE=${LOGFILE:-/var/log/$CMD.log}
LOGTAG=${LOGTAG:-$CMD}
LOGUSER=${LOGUSER:-root}
LOGGROUP=${LOGGROUP:-adm}
LOGMODE=${LOGMODE:-644}
LOGSYSLOG=${LOGSYSLOG:-true}
LOGGERPRIO=${LOGGERPRIO:-user.notice}
LOGGERARGS=${LOGGERARGS:-}
LOGROTATIONS=${LOGROTATIONS:-7}
VERBOSE=${VERBOSE:-false}
KU_LOG_SAVEIFS=$IFS

ku_cap_logfile()
{
	local size=${LOGSIZE:-10240}
	local file=${LOGFILE:-}
	local user=${LOGUSER:-}
	local group=${LOGGROUP:-}
	local mode=${LOGMODE:-}
	local rotations=${LOGROTATIONS:-}
	local actual_size=

	[ "X$file" = "X" ] && return 0

	[ -f "$file" ] || {
		echo -n >"$file" || {
			echo "can't create '$file'" >&2
			return 1
		}
		[ "X$user" != "X" ] && {
			getent passwd $user >/dev/null 2>&1 && {
				chown $user "$file" 2>/dev/null || :
			}
		}
		[ "X$group" != "X" ] && {
			getent group $group >/dev/null 2>&1 && {
				chgrp $group "$file" 2>/dev/null || :
			}
		}
		[ "X$mode" != "X" ] && {
			chmod $mode "$file" 2>/dev/null || :
		}
	}
	echo -n >>"$file" || {
		echo "can't write on '$file'" >&2
		return 1
	}
	actual_size=$(( $(stat -c '%s' "$file") / 1024 ))
	[ $actual_size -gt $size ] && {
		/usr/bin/savelog -c $rotations -t -p "$file"
	}
	return 0
}


ku_log()
{
	local pid=${PID:-}
	local tag=${LOGTAG:-}
	local syslog=${LOGSYSLOG:-false}
	local logfile=${LOGFILE:-}
	local verbose=${VERBOSE:-false}
	local priority=
	local save_ifs=$IFS

	# restore standard IFS, just in case some script have changed it
	# to avoid shell args interpretation errors
	IFS=$KU_LOG_SAVEIFS

	tag="-t ${tag:-$(id -u -n)}"

	[ "X$pid" != "X" ] && {
		tag="${tag}-P$pid"
		pid="P$pid "
	}

	[ "X${LOGGERPRIO:-}" != "X" ]	&& priority="-p $LOGGERPRIO"
	[ "X$verbose" = "Xtrue" ]	&& echo -e "$@" >&2
	[ "X$logfile" != "X" ]		&& echo -e "$(date '+%Y%m%d %H%M%S') $pid$*" >>$logfile
	$syslog && /usr/bin/logger ${LOGGERARGS:-} $priority $tag "$*"

	IFS=$save_ifs
	return 0
}

ku_loglines()
{
	local line=

	if [ $# != 0 ]
	then
		echo -e "$@" | sed -e 's/^/x/' | while read line
		do
			line=$(echo "$line" | sed -e 's/^x//')
			ku_log "$line"
		done
	else
		sed -e 's/^/x/' | while read line
		do
			line=$(echo "$line" | sed -e 's/^x//')
			ku_log "$line"
		done
	fi
	return 0
}
