#!/bin/bash
#
# lock.sh - bash functions to manage locking
#
# __copy1__
# __copy2__
#
# v1.14 (2022-03-09)
#
export KU_LOCKFILE


## ku_lock [file]
##
## activate the lock, writing che currend PID ($$) into
## the file $KU_LOCKFILE, returns 0 if the lock is acquired,
## 1 otherwise
##
## if 'file' parm is present, is used and KU_LOCKFILE
## is set
##
## if there already are a lock active using this file
## (see ku_lock_is_active function), an error is returned
## with proper message on stdout
##
ku_lock()
{
	local out=

	KU_LOCKFILE=${KU_LOCKFILE:-""}

	[ $# != 0 ] && KU_LOCKFILE="$1"

	[ "X$KU_LOCKFILE" == "X" ] && {
		echo "error, you must set \$KU_LOCKFILE or it as argument" >&2
		return 1
	}

	ku_lock_is_active && {
		echo "lock already active on $KU_LOCKFILE"
		return 1
	}
	out=$(echo $$ >"$KU_LOCKFILE" 2>&1) || {
		echo "lock failed on $KU_LOCKFILE: $out"
		return 1
	}
	return 0
}

## ku_lock_remove [file]
##
## remove the current lock (remove the file $KU_LOCKFILE)
##
## if 'file' parm is present, is used and KU_LOCKFILE
## is set
##
ku_lock_remove()
{
	local out=

	KU_LOCKFILE=${KU_LOCKFILE:-""}

	[ $# != 0 ] && KU_LOCKFILE="$1"

	[ -e "$KU_LOCKFILE" ] || return 0	# not present, ok

	out=$(/bin/rm "$KU_LOCKFILE" 2>&1) || {
		echo "lock remove failed: $out"
		return 1
	}
	return 0
}


## ku_lock_is_active [file]
##
## check if a lock is active, baed on the file $KU_LOCKFILE,
## returns 0 if active, 1 if not
##
## if 'file' parm is present, is used and KU_LOCKFILE
## is set
##
## a lock is still active if 1) the file exists, and 2) the
## process with the pid recorded in the file is still running
##
## if the process is dead, a stale lock condition is detected,
## the lockfile is removed and the functions returns 1
##
## if any strange condition is detected (the lockfile is not
## a file, or contains unexpected data, the file cannot be read,
## etc), we use a prudent approach and the lock is considered
## active
##
ku_lock_is_active()
{
	local out=
	local pid=

	KU_LOCKFILE=${KU_LOCKFILE:-""}

	[ $# != 0 ] && KU_LOCKFILE="$1"

	[ -e "$KU_LOCKFILE" ] || return 1	# not present, no lock
	[ -f "$KU_LOCKFILE" ] || return 0	# ?? not a file, active
	
	out=$(cat "$KU_LOCKFILE" 2>&1) || {
		echo "lock read error on $KU_LOCKFILE: $out" >&2
		return 0
	}
	pid=$(cat "$KU_LOCKFILE")
	echo "$pid" | grep -q "^[0-9][0-9]*$" || {
		echo "lockfile $KU_LOCKFILE contains unknown data, lock assumed" >&2
		return 0	# locked
	}
	#out=$(ps | grep "^$pid ")
	#[ "$out" == "" ] && {
	[ -d /proc/$pid ] || {
		echo "stale lock detected, lock removed" >&2
		ku_lock_remove
		return 1	# no lock
	}
	return 0		# still locked
}
