#!/bin/bash
#
# echo.sh - bash functions to manage fancy output messages
#
# __copy1__
# __copy2__
#
## echocl string(s)
## echocr string(s)
##
## echoes 'string(s)' after clearing current line up to end-of-line;
## a newline is not added at end, add it using "\n" if needed
##
## echocr reposition cursor to start of line, echocl not (working
## like regular 'echo -n' or 'printf' command)
##
## the purpouse of these function is to produce fancy interactive
## display, eg. for progress informations, DO NOT USE them to give
## important status messages, because they don't use standard output
## and error channels
##
## the output goes directly on tty, bypassing regular redirectrion
## of standard output and error; set VERBOSE=false to disable output
##
## those functions are enabled only if standard input is a regular tty,
## (pts) and if terminfo cap 'el' (clear-end-of-line) is defined,
## otherwise are disabled (dummy)
##
## ENVIRONMENT
## those vars are defined and exported with proper values, or empty
## if stdin is not a tty:
##
##  KU_TTY	current tty
##  KU_CLEAR	terminfo sequence for whole screen clear (clear)
##  KU_CLEOL	terminfo sequence for clear-to-end-of-line (el)
##  KU_BOLD	terminfo sequence for bold text (bold)
##  KU_REV	terminfo sequence for reverse text (rev)
##  KU_NORM	terminfo sequence for normal text (sgr0)
##
## OTHER FUNCTIONS
##
## echotrunc len string
##
##   echotrunc echoes 'string', truncating if it's len > 'len'; the
##   truncation will be done in the middle of string, so preserving
##   start and end, and replacing the removed text with four dots
##   '....'
##
#
export KU_TTY=$(ls -l /proc/$$/fd/0 2>/dev/null) || :
export KU_CLEAR=
export KU_CLEOL=
export KU_BOLD=
export KU_REV=
export KU_NORM=
export VERBOSE=${VERBOSE:-"true"}
export TERM=${TERM:-dumb}


# disable fancy tty output if ...
#
# stdout is not a tty
#
if [ ! -t 1 ]
then
	KU_TTY=
fi

if [ "$KU_TTY" != "" ]
then
	#
	# or tput doesn't works
	#
	if ! tput el >/dev/null 2>&1
	then
		KU_TTY=
	fi
fi

if [ "$KU_TTY" != "" ]
then
	if echo "$KU_TTY" | grep -q ' -> '
	then
		KU_TTY=$(echo "$KU_TTY" | sed -e 's/.*-> *//')
		[ -w "$KU_TTY" ] || KU_TTY=
	fi
fi


if [ "$KU_TTY" != "" ]
then
	KU_CLEAR=$(tput clear 2>/dev/null)	|| :
	KU_CLEOL=$(tput el 2>/dev/null)		|| :
	KU_BOLD=$(tput bold 2>/dev/null)	|| :
	KU_REV=$(tput rev 2>/dev/null)		|| :
	KU_NORM=$(tput sgr0 2>/dev/null)	|| :
	echocl() { $VERBOSE && echo -en "\r${KU_CLEOL}$*" >>"$KU_TTY"; return 0; }
	echocr() { $VERBOSE && echo -en "\r${KU_CLEOL}$*\r" >>"$KU_TTY" ; return 0; }
else
	echocl() { $VERBOSE && echo -en "\r$*"; return 0; }
	echocr() { $VERBOSE && echo -en "\r$*\r"; return 0; }
fi



echotrunc()
{
	local len=$1; shift
	local str="$@"
	local out="$str"
	local mid=$((($len/2)-2))

	if [ ${#str} -gt $len ]
	then
		out=${str:0:$mid}"...."${str: -$mid}
	fi
	echo "$out"
}
